// Utilities
const MANIFEST_URL_RELATIVE = './search/file-manifest.json';
const MANIFEST_CACHE_KEY = 'fileBrowser.manifest';
const MANIFEST_META_KEY = 'fileBrowser.manifestMeta';

function loadManifestFromCache() {
  try {
    const metaRaw = localStorage.getItem(MANIFEST_META_KEY);
    const dataRaw = localStorage.getItem(MANIFEST_CACHE_KEY);
    if (!metaRaw || !dataRaw) return null;
    const meta = JSON.parse(metaRaw);
    const data = JSON.parse(dataRaw);
    if (!meta || !data) return null;
    // Ensure cache matches structure minimally
    if (!data.sections || !Array.isArray(data.sections)) return null;
    return { meta, data };
  } catch (_) { return null; }
}

function saveManifestToCache(manifest) {
  try {
    const meta = { generated: manifest.generated, cachedAt: new Date().toISOString() };
    localStorage.setItem(MANIFEST_META_KEY, JSON.stringify(meta));
    localStorage.setItem(MANIFEST_CACHE_KEY, JSON.stringify(manifest));
  } catch (_) { /* ignore quota or private mode errors */ }
}

async function fetchManifestOverHttp() {
  // Prefer relative URL so it works whether hosted or opened from nested folders
  const res = await fetch(MANIFEST_URL_RELATIVE, { cache: 'force-cache' });
  if (!res.ok) throw new Error('Failed to load manifest');
  return res.json();
}

async function loadManifest() {
  // 1) Try cache
  const cached = loadManifestFromCache();
  if (cached?.data) return cached.data;
  
  // 2) If running from file://, check for pre-loaded manifest first
  const isFileProtocol = typeof location !== 'undefined' && location.protocol === 'file:';
  if (isFileProtocol && typeof window !== 'undefined' && window.FILE_MANIFEST) {
    const manifest = window.FILE_MANIFEST;
    saveManifestToCache(manifest);
    return manifest;
  }
  
  // 3) Try HTTP fetch
  try {
    const manifest = await fetchManifestOverHttp();
    saveManifestToCache(manifest);
    return manifest;
  } catch (err) {
    // 4) If running from file://, attempt to load JS fallback automatically
    if (isFileProtocol) {
      const loaded = await tryLoadManifestViaScriptFallback();
      if (loaded) return loaded;
    }
    // 5) Fallback: use already-present global if available
    if (typeof window !== 'undefined' && window.FILE_MANIFEST) {
      const manifest = window.FILE_MANIFEST;
      saveManifestToCache(manifest);
      return manifest;
    }
    throw err;
  }
}

function tryLoadManifestViaScriptFallback() {
  return new Promise((resolve) => {
    if (typeof window === 'undefined') return resolve(null);
    if (window.FILE_MANIFEST) return resolve(window.FILE_MANIFEST);
    const script = document.createElement('script');
    script.src = './search/file-manifest.js';
    script.async = true;
    script.onload = () => {
      if (window.FILE_MANIFEST) {
        try { saveManifestToCache(window.FILE_MANIFEST); } catch (_) {}
        resolve(window.FILE_MANIFEST);
      } else { resolve(null); }
    };
    script.onerror = () => resolve(null);
    document.head.appendChild(script);
  });
}

// ---------- Icons (inline SVG) ----------
function svgFolder() {
  return '<svg class="icon" viewBox="0 0 24 24" fill="#9ecfff" xmlns="http://www.w3.org/2000/svg"><path d="M10 4l2 2h8a2 2 0 012 2v9a3 3 0 01-3 3H7a3 3 0 01-3-3V7a3 3 0 013-3h3z" fill="#2b6cb0"/><path d="M3 9h18v8a2 2 0 01-2 2H5a2 2 0 01-2-2V9z" fill="#3182ce"/></svg>';
}
function svgFileGeneric() {
  return '<svg class="icon" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="#e2e8f0" d="M6 2h9l5 5v13a2 2 0 01-2 2H6a2 2 0 01-2-2V4a2 2 0 012-2z"/><path fill="#94a3b8" d="M15 2v5h5"/></svg>';
}
function svgForExt(ext) {
  const e = (ext||'').toLowerCase();
  const color = (e==='pdf')?'#dc2626':(e==='mp4'||e==='mov')?'#0ea5e9':(e==='mp3'||e==='wav')?'#22c55e':(e==='jpg'||e==='jpeg'||e==='png'||e==='gif')?'#f59e0b':(e==='zip'||e==='rar')?'#7c3aed':(e==='doc'||e==='docx')?'#1d4ed8':(e==='xls'||e==='xlsx'||e==='csv')?'#16a34a':(e==='ppt'||e==='pptx')?'#ea580c':'#64748b';
  return `<svg class="icon" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="#e2e8f0" d="M6 2h9l5 5v13a2 2 0 01-2 2H6a2 2 0 01-2-2V4a2 2 0 012-2z"/><path fill="#94a3b8" d="M15 2v5h5"/><rect x="7" y="15" width="10" height="4" rx="2" fill="${color}"/></svg>`;
}

// ---------- Tree building from manifest paths ----------
function normalizePath(path) {
  return (path||'').replace(/^\.+\//,'').replace(/^\/+/, '');
}
function splitPath(path) {
  const cleaned = normalizePath(path);
  return cleaned.split('/').filter(Boolean);
}

function createFolderNode(name, pathSegments) {
  return { kind: 'folder', name, pathSegments: [...pathSegments], children: [], size: 0, fileCount: 0 };
}
function createFileNode(file, pathSegments) {
  return { kind: 'file', name: file.name, pathSegments: [...pathSegments], size: file.size||0, modified: file.modified||'', type: file.type||'', manifestPath: file.path };
}

function buildVirtualTree(manifest) {
  const root = createFolderNode('Root Files', []);
  
  // Helper function to find or create a folder node by path segments
  const findOrCreateFolderByPath = (pathSegments) => {
    let current = root;
    pathSegments.forEach((segment) => {
      let child = current.children.find(c => c.kind === 'folder' && c.name === segment);
      if (!child) {
        child = createFolderNode(segment, current.pathSegments.concat(segment));
        current.children.push(child);
      }
      current = child;
    });
    return current;
  };
  
  // Process each section from the manifest
  (manifest.sections || []).forEach((section) => {
    // Create the section folder
    const sectionPath = splitPath(section.path || '');
    const sectionFolder = findOrCreateFolderByPath(sectionPath);
    
    // Recursively process nested folders structure
    const processFolder = (folderData, parentNode) => {
      // Create this folder node
      const folderNode = createFolderNode(folderData.name, parentNode.pathSegments.concat(folderData.name));
      parentNode.children.push(folderNode);
      
      // Process files in this folder
      (folderData.files || []).forEach((file) => {
        const fileNode = createFileNode(file, folderNode.pathSegments.concat(file.name));
        folderNode.children.push(fileNode);
      });
      
      // Recursively process subfolders
      (folderData.folders || []).forEach((subfolder) => {
        processFolder(subfolder, folderNode);
      });
    };
    
    // Process the nested folders structure from the manifest
    (section.folders || []).forEach((folder) => {
      processFolder(folder, sectionFolder);
    });
    
    // Process files directly in the section
    (section.files || []).forEach((file) => {
      const fileNode = createFileNode(file, sectionFolder.pathSegments.concat(file.name));
      sectionFolder.children.push(fileNode);
    });
  });

  // Compute aggregates (size, file count) and sort
  function compute(node) {
    if (node.kind === 'file') { 
      return { size: node.size || 0, files: 1 }; 
    }
    
    let size = 0, files = 0;
    node.children.forEach((child) => {
      const result = compute(child);
      size += result.size;
      files += result.files;
    });
    
    node.size = size;
    node.fileCount = files;
    
    // Sort: folders first, then files, both alphabetically
    node.children.sort((a, b) => {
      if (a.kind !== b.kind) return a.kind === 'folder' ? -1 : 1;
      return a.name.localeCompare(b.name, undefined, { sensitivity: 'base' });
    });
    
    return { size, files };
  }
  
  compute(root);
  return root;
}

function pathToKey(pathSegments){ return '/' + pathSegments.join('/'); }

// ---------- Persistent UI state ----------
const EXPANDED_KEY = 'fileBrowser.expandedPaths';
const SELECTED_KEY = 'fileBrowser.routePath';
const CURRENT_PATH_KEY = 'fileBrowser.currentPath';
function loadExpanded(){ try{ return new Set(JSON.parse(localStorage.getItem(EXPANDED_KEY)||'[]')); }catch(_){ return new Set(); } }
function saveExpanded(set){ try{ localStorage.setItem(EXPANDED_KEY, JSON.stringify(Array.from(set))); }catch(_){} }
function saveSelected(path){ try{ localStorage.setItem(SELECTED_KEY, JSON.stringify(path)); }catch(_){} }
function loadSelected(){ try{ return JSON.parse(localStorage.getItem(SELECTED_KEY)||'null'); }catch(_){ return null; } }
function saveCurrentPath(path){ try{ localStorage.setItem(CURRENT_PATH_KEY, JSON.stringify(path)); }catch(_){} }
function loadCurrentPath(){ try{ return JSON.parse(localStorage.getItem(CURRENT_PATH_KEY)||'[]'); }catch(_){ return []; } }

// ---------- Renderers ----------
function renderTree(container, rootNode, onSelect, expandedPaths, selectedPath) {
  container.innerHTML = '';
  const ul = document.createElement('ul');
  ul.className = 'tree';

  function renderFolder(node, parentUl){
    const li = document.createElement('li');
    const row = document.createElement('div');
    row.className = 'item';
    
    const itemPath = pathToKey(node.pathSegments);
    const isExpanded = EXPANDED_PATHS.has(itemPath);
    const toggle = document.createElement('span');
    toggle.className = 'folder-toggle';
    toggle.textContent = isExpanded ? '▼' : '▶';
    
    const iconSpan = document.createElement('span'); 
    iconSpan.className = 'folder-icon';
    iconSpan.innerHTML = svgFolder();
    
    const label = document.createElement('span'); 
    label.className = 'folder-name';
    label.textContent = node.name;
    
    row.appendChild(toggle);
    row.appendChild(iconSpan);
    row.appendChild(label);
    
    if (selectedPath && itemPath===pathToKey(selectedPath)) row.classList.add('selected');
    li.appendChild(row);

    const inner = document.createElement('ul');
    inner.className = 'tree';
    inner.style.marginLeft = '20px';
    inner.style.display = isExpanded ? 'block' : 'none';

    // Expand/collapse when clicking the toggle arrow
    toggle.addEventListener('click', (e) => {
      e.stopPropagation();
      const key = pathToKey(node.pathSegments);
      if (EXPANDED_PATHS.has(key)) {
        EXPANDED_PATHS.delete(key);
      } else {
        EXPANDED_PATHS.add(key);
      }
      saveExpanded(EXPANDED_PATHS);
      refreshTreeView();
    });

    // Navigate when clicking the folder name
    label.addEventListener('click', (e) => {
      e.stopPropagation();
      navigateToBreadcrumbLevel(node.pathSegments);
    });

    node.children.forEach((child)=>{
      if (child.kind==='folder') renderFolder(child, inner);
      else {
        const fli = document.createElement('li');
        const frow = document.createElement('div'); 
        frow.className='item file';
        
        const fileIcon = document.createElement('span'); 
        fileIcon.className='file-icon'; 
        fileIcon.innerHTML = svgForExt(getFileExtension(child.name));
        
        const fileName = document.createElement('span'); 
        fileName.className='file-name';
        fileName.textContent = child.name;
        
        const fileSize = document.createElement('span');
        fileSize.className = 'file-size';
        fileSize.textContent = formatFileSize(child.size || 0);
        
        const downloadBtn = document.createElement('button');
        downloadBtn.className = 'download-btn';
        downloadBtn.textContent = '↓';
        downloadBtn.title = 'Download file';
        
        frow.appendChild(fileIcon); 
        frow.appendChild(fileName);
        frow.appendChild(fileSize);
        frow.appendChild(downloadBtn);
        
        // File click - open in new tab
        frow.addEventListener('click', (e) => {
          if (e.target === downloadBtn) return;
          openFile(child);
        });
        
        // Download button click
        downloadBtn.addEventListener('click', (e) => {
          e.stopPropagation();
          downloadFile(child);
        });
        
        fli.appendChild(frow); 
        inner.appendChild(fli);
      }
    });
    li.appendChild(inner); parentUl.appendChild(li);
  }

  // Render top-level children of root
  rootNode.children.filter(c=>c.kind==='folder').forEach((folder)=> renderFolder(folder, ul));
  // Also render root files (if any)
  rootNode.children.filter(c=>c.kind==='file').forEach((file)=>{
    const fli = document.createElement('li');
    const frow = document.createElement('div'); 
    frow.className='item file';
    
    const fileIcon = document.createElement('span'); 
    fileIcon.className='file-icon'; 
    fileIcon.innerHTML = svgForExt(getFileExtension(file.name));
    
    const fileName = document.createElement('span'); 
    fileName.className='file-name';
    fileName.textContent = file.name;
    
    const fileSize = document.createElement('span');
    fileSize.className = 'file-size';
    fileSize.textContent = formatFileSize(file.size || 0);
    
    const downloadBtn = document.createElement('button');
    downloadBtn.className = 'download-btn';
    downloadBtn.textContent = '↓';
    downloadBtn.title = 'Download file';
    
    frow.appendChild(fileIcon); 
    frow.appendChild(fileName);
    frow.appendChild(fileSize);
    frow.appendChild(downloadBtn);
    
    // File click - open in new tab
    frow.addEventListener('click', (e) => {
      if (e.target === downloadBtn) return;
      openFile(file);
    });
    
    // Download button click
    downloadBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      downloadFile(file);
    });
    
    fli.appendChild(frow); 
    ul.appendChild(fli);
  });

  container.appendChild(ul);
}

function manifestPathToUrl(manifestPath){
  const cleaned = (manifestPath || '').replace(/^\.+\//,'').replace(/^\/+/, '');
  return '../../' + cleaned;
}

// Helper functions for file operations
function formatFileSize(bytes) {
  if (bytes === 0) return '0 B';
  const k = 1024;
  const sizes = ['B', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
}

function openFile(fileNode) {
  if (!fileNode.manifestPath) return;
  const url = manifestPathToUrl(fileNode.manifestPath);
  const size = fileNode.size || 0;
  
  // Warn for large files
  if (size > 100 * 1024 * 1024) { // 100MB
    const sizeMB = (size / 1024 / 1024).toFixed(1);
    const confirmed = confirm(`This file is ${sizeMB} MB and may take time to load. Continue?`);
    if (!confirmed) return;
  }
  
  window.open(url, '_blank', 'noopener,noreferrer');
}

function downloadFile(fileNode) {
  if (!fileNode.manifestPath) return;
  const url = manifestPathToUrl(fileNode.manifestPath);
  const size = fileNode.size || 0;
  
  // Warn for very large files
  if (size > 500 * 1024 * 1024) { // 500MB
    const sizeMB = (size / 1024 / 1024).toFixed(1);
    const confirmed = confirm(`This file is ${sizeMB} MB and may take significant time to download. Continue?`);
    if (!confirmed) return;
  }
  
  // Create temporary download link
  const a = document.createElement('a');
  a.href = url;
  a.download = fileNode.name;
  a.style.display = 'none';
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
}

// Right panel functionality removed - using simplified tree-only approach

// ---------- Enhanced Clickable Breadcrumbs ----------
function renderBreadcrumbs(barEl) {
  // Show current navigation path instead of deepest expanded folder
  const currentPath = CURRENT_PATH || [];
  
  let html = '<a href="#" class="breadcrumb-home">📁 Browse Files</a>';
  
  if (currentPath.length > 0) {
    html += '<span class="sep"> / </span>';
    currentPath.forEach((seg, idx) => {
      if (idx > 0) html += '<span class="sep"> / </span>';
      // Make each segment clickable by creating a link
      html += `<a href="#" class="breadcrumb-folder" data-path-segments="${currentPath.slice(0, idx + 1).join('|')}">${seg}</a>`;
    });
  }
  
  barEl.innerHTML = html;
  
  // Home link resets view
  const homeLink = barEl.querySelector('.breadcrumb-home');
  if (homeLink) {
    homeLink.addEventListener('click', (e) => {
      e.preventDefault();
      navigateToBreadcrumbLevel([]);
    });
  }
  
  // Add click handlers for breadcrumb folder segments
  const folderLinks = barEl.querySelectorAll('.breadcrumb-folder');
  folderLinks.forEach(link => {
    link.addEventListener('click', (e) => {
      e.preventDefault();
      const pathSegments = link.getAttribute('data-path-segments').split('|').filter(Boolean);
      navigateToBreadcrumbLevel(pathSegments);
    });
  });
}

// Function to navigate to a specific breadcrumb level
function navigateToBreadcrumbLevel(targetPathSegments) {
  // Set the current path to the target
  CURRENT_PATH = [...targetPathSegments];
  
  // Clear existing expanded paths
  EXPANDED_PATHS.clear();
  
  // If navigating to root, just clear everything
  if (targetPathSegments.length === 0) {
    FILTER_QUERY = '';
    FILTER_TYPE = '';
    
    // Reset UI elements
    const searchInput = document.getElementById('search-input');
    const typeFilter = document.querySelector('.type-filter');
    const searchClearBtn = document.getElementById('search-clear-btn');
    if (searchInput) searchInput.value = '';
    if (typeFilter) typeFilter.value = '';
    if (searchClearBtn) searchClearBtn.classList.remove('active');
    
    saveExpanded(EXPANDED_PATHS);
    saveCurrentPath(CURRENT_PATH);
    
    // Set flag to prevent folder click handlers from overriding CURRENT_PATH
    IS_PROGRAMMATIC_RENDER = true;
    refreshTreeView();
    // Reset flag after render completes
    setTimeout(() => { IS_PROGRAMMATIC_RENDER = false; }, 0);
    return;
  }
  
  // Expand path up to and including the target level
  for (let i = 1; i <= targetPathSegments.length; i++) {
    const pathKey = '/' + targetPathSegments.slice(0, i).join('/');
    EXPANDED_PATHS.add(pathKey);
  }
  
  saveExpanded(EXPANDED_PATHS);
  saveCurrentPath(CURRENT_PATH);
  
  // Set flag to prevent folder click handlers from overriding CURRENT_PATH
  IS_PROGRAMMATIC_RENDER = true;
  refreshTreeView();
  // Reset flag after render completes
  setTimeout(() => { IS_PROGRAMMATIC_RENDER = false; }, 0);
}
// Simplified node finding (keep for search functionality)
function findNodeByPath(root, pathSegments){
  let node = root;
  for (const seg of pathSegments){
    node = node.children.find(c=>c.kind==='folder' && c.name===seg);
    if (!node) return null;
  }
  return node;
}

// Global state - simplified
let GLOBAL_TREE_ROOT = null;
let FILTER_QUERY = '';
let FILTER_TYPE = '';
let EXPANDED_PATHS = new Set();
let CURRENT_PATH = []; // Track the user's current navigation context
let IS_PROGRAMMATIC_RENDER = false; // Flag to prevent conflicts during tree re-rendering

// Simple refresh function for tree view
function refreshTreeView() {
  if (!GLOBAL_TREE_ROOT) return;
  const tree = document.getElementById('folder-tree');
  const crumbsBar = document.getElementById('breadcrumbs-bar');
  
  if (tree && crumbsBar) {
    const rootForView = getFilteredRoot();
    renderTree(tree, rootForView, () => {}, EXPANDED_PATHS, []);
    renderBreadcrumbs(crumbsBar);
  }
}

// Search filtering
function fileMatches(node) {
  if (node.kind !== 'file') return false;
  const q = FILTER_QUERY.trim().toLowerCase();
  const type = (FILTER_TYPE||'').toLowerCase();
  const nameMatch = !q || node.name.toLowerCase().includes(q);
  const ext = getFileExtension(node.name);
  const typeMatch = !type || ext === type;
  return nameMatch && typeMatch;
}

function clonePruned(node) {
  if (node.kind === 'file') {
    return fileMatches(node) ? { ...node } : null;
  }
  const cloned = { ...node, children: [] };
  node.children.forEach((child) => {
    const c = clonePruned(child);
    if (c) cloned.children.push(c);
  });
  if (cloned.children.length === 0) return null;
  return cloned;
}

function getFilteredRoot() {
  if (!FILTER_QUERY && !FILTER_TYPE) return GLOBAL_TREE_ROOT;
  const pruned = clonePruned(GLOBAL_TREE_ROOT);
  return pruned || createFolderNode('No results found', []);
}





// Helper functions
function getFileExtension(name) {
  const idx = name.lastIndexOf('.');
  return idx >= 0 ? name.slice(idx + 1).toLowerCase() : '';
}

function buildTypeSet(manifest) {
  const set = new Set();
  
  // Helper function to recursively process folders
  const processFolder = (folder) => {
    // Process files in this folder
    (folder.files || []).forEach((f) => {
      const ext = getFileExtension(f.name);
      if (ext) set.add(ext.toLowerCase());
    });
    
    // Recursively process subfolders
    (folder.folders || []).forEach((subfolder) => {
      processFolder(subfolder);
    });
  };
  
  (manifest.sections || []).forEach((section) => {
    // Process files directly in the section
    (section.files || []).forEach((f) => {
      const ext = getFileExtension(f.name);
      if (ext) set.add(ext.toLowerCase());
    });
    
    // Process nested folders
    (section.folders || []).forEach((folder) => {
      processFolder(folder);
    });
  });
  
  return Array.from(set).sort();
}

function getFileTypeFriendlyName(extension) {
  const friendlyNames = {
    'pdf': 'Adobe Acrobat (PDF)',
    'docx': 'Microsoft Word (DOCX)',
    'doc': 'Microsoft Word (DOC)', 
    'txt': 'Text Document (TXT)',
    'mp4': 'Video File (MP4)',
    'mp3': 'Audio File (MP3)',
    'wav': 'Audio File (WAV)',
    'jpg': 'Image File (JPG)',
    'jpeg': 'Image File (JPEG)',
    'png': 'Image File (PNG)',
    'gif': 'Image File (GIF)',
    'zip': 'Compressed Archive (ZIP)',
    'rar': 'Compressed Archive (RAR)',
    'pptx': 'Microsoft PowerPoint (PPTX)',
    'ppt': 'Microsoft PowerPoint (PPT)',
    'xlsx': 'Microsoft Excel (XLSX)',
    'xls': 'Microsoft Excel (XLS)',
    'csv': 'Spreadsheet (CSV)'
  };
  
  return friendlyNames[extension.toLowerCase()] || `${extension.toUpperCase()} File (${extension.toUpperCase()})`;
}

function renderLoadError(details, tree, reason) {
  const help = document.createElement('div');
  help.innerHTML = `
    <div style="color:#ddd; line-height:1.5">
      <h3 style="margin:0 0 8px">We couldn't load the file index yet</h3>
      <p style="margin:0 0 8px; color:#ccc">If you're opening this from a USB or local folder, your browser may block JSON loading on file:// URLs.</p>
      <ul style="margin:0 0 8px 1.2rem; color:#ccc">
        <li>Try the fallback loader (no internet required)</li>
        <li>Or open this site via a local/web server</li>
        <li>Or host on S3/CDN (works out-of-the-box)</li>
      </ul>
      <div style="display:flex; gap:8px; margin-top:8px">
        <button class="btn-retry" style="padding:6px 10px">Retry</button>
        <button class="btn-fallback" style="padding:6px 10px">Try fallback</button>
      </div>
      <div style="margin-top:6px; color:#777; font-size:12px">${reason ? String(reason) : ''}</div>
    </div>
  `;
  if (details) details.innerHTML = '';
  if (details) details.appendChild(help);
  if (tree) tree.innerHTML = '<div class="spinner">Index not loaded</div>';
  return help;
}

// Simplified initialization
document.addEventListener('DOMContentLoaded', async () => {
  const tree = document.getElementById('folder-tree');
  const input = document.getElementById('search-input');
  const filtersHost = document.getElementById('search-filters');
  const crumbsBar = document.getElementById('breadcrumbs-bar');

  // Show loading state
  if (tree) tree.innerHTML = '<div class="spinner">Loading file index…</div>';
  if (input) input.disabled = true;

  try {
    // Load manifest and build tree
    const manifest = await loadManifest();
    GLOBAL_TREE_ROOT = buildVirtualTree(manifest);
    EXPANDED_PATHS = loadExpanded();
    CURRENT_PATH = loadCurrentPath();

    // Build type filter dropdown
    const types = buildTypeSet(manifest);
    if (filtersHost) {
      const select = document.createElement('select');
      select.setAttribute('aria-label', 'Filter by file type');
      select.className = 'type-filter';
      select.id = 'type-filter-select';
      const allOpt = document.createElement('option');
      allOpt.value = '';
      allOpt.textContent = 'File Types';
      select.appendChild(allOpt);
      types.forEach((t) => {
        const opt = document.createElement('option');
        opt.value = t.toLowerCase();
        opt.textContent = getFileTypeFriendlyName(t);
        select.appendChild(opt);
      });
      select.addEventListener('change', () => {
        FILTER_TYPE = select.value;
        refreshTreeView();
      });
      filtersHost.appendChild(select);
    }

    // Search input with debouncing and clear button
    const searchClearBtn = document.getElementById('search-clear-btn');
    const resetViewBtn = document.getElementById('reset-view-btn');
    
    if (input) {
      input.disabled = false;
      let searchTimeout = null;
      
      // Update search and show/hide clear button
      const updateSearch = () => {
        FILTER_QUERY = input.value || '';
        if (FILTER_QUERY) {
          searchClearBtn.classList.add('active');
        } else {
          searchClearBtn.classList.remove('active');
        }
        refreshTreeView();
      };
      
      input.addEventListener('input', () => {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(updateSearch, 300);
      });
      
      // Clear search button
      if (searchClearBtn) {
        searchClearBtn.addEventListener('click', () => {
          input.value = '';
          FILTER_QUERY = '';
          searchClearBtn.classList.remove('active');
          refreshTreeView();
          input.focus();
        });
      }
    }
    
    // Reset view button - clears both search and filter
    if (resetViewBtn) {
      resetViewBtn.addEventListener('click', () => {
        // Clear search
        if (input) {
          input.value = '';
          FILTER_QUERY = '';
          searchClearBtn.classList.remove('active');
        }
        
        // Clear filter
        const typeSelect = document.getElementById('type-filter-select');
        if (typeSelect) {
          typeSelect.value = '';
          FILTER_TYPE = '';
        }
        
        // Clear expanded paths and current path to reset breadcrumbs
        EXPANDED_PATHS.clear();
        CURRENT_PATH = [];
        saveExpanded(EXPANDED_PATHS);
        saveCurrentPath(CURRENT_PATH);
        
        refreshTreeView();
      });
    }

    // Initial render
    refreshTreeView();

  } catch (err) {
    // Show error with retry options
    const errorPanel = renderLoadError(null, tree, err?.message || '');
    const retryBtn = errorPanel.querySelector('.btn-retry');
    const fallbackBtn = errorPanel.querySelector('.btn-fallback');
    
    retryBtn?.addEventListener('click', () => {
      document.location.reload();
    });
    
    fallbackBtn?.addEventListener('click', async () => {
      errorPanel.innerHTML = '<div class="spinner">Loading fallback…</div>';
      try {
        const manifest = await tryLoadManifestViaScriptFallback();
        if (!manifest) throw new Error('Fallback not available');
        saveManifestToCache(manifest);
        document.location.reload();
      } catch (_) {
        errorPanel.innerHTML = '<div style="color:#f66">Fallback failed. Please try hosting this site via a web server.</div>';
      }
    });
    
    console.error('File browser initialization failed:', err);
  }
});


